/******************************************\
 *                                        *
 *       read a configuration file        *
 *                                        *
 *   Copyright (C) 6/1994 Klaus Bechert   *
 *                                        *
\******************************************/


/************************************************************/
/************ part I: a list off all variables **************/

/*
 * if our list contains variables of 'double' type...
 */
#define floating_point   1


/*
 * the configuration variables which should be all public
 */
char *fontname = "pc437p10";
int tabstop = 8;
int columns = 1;
int linesize = 80;
double hsize = 6.5;
double vsize = 8.9;
double hoffset = 0.0;
double voffset = 0.0;
double widthcolumn = 6.5;


/*
 * reference to all the configuration variables
 */
static struct {char *idname; void *ref; int  type;} list[]=
{
 {"fontname", &fontname, 's'},
 {"tabstop", &tabstop, 'i'},
 {"columns",  &columns, 'i'},
 {"linesize", &linesize, 'i'},
 {"hsize", &hsize, 'd'},
 {"vsize", &vsize, 'd'},
 {"hoffset", &hoffset, 'd'},
 {"voffset", &voffset, 'd'},
 {"widthcolumn", &widthcolumn, 'd'},
 {0,0,0}
};


/************************************************************/
/************** part II: read the cfg file ******************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>


/*
 * extensions of ctype
 */
#define iz(name, x) ((c>=0 && c<=127) ? is##name(c) : x)
#define izalpha(c) iz(alpha,1)
#define izalnum(c) iz(alnum,1)
#define izspace(c) iz(space,0)


/*
 * globals but not public
 */
#define MAXLEN 127
static char idname[MAXLEN+1];       /* buffer for variable name */
static char value[MAXLEN+1];        /* buffer for value */
static char *filename;              /* pointer to filename */
static int  line;                   /* current line number */


/*
 * error handler
 */
static void error(char *s)
{
 fprintf(stderr, "*** error in %s line %d ***\n", filename, line);
 fprintf(stderr, "*** %s ***\n", s);
}


/*
 * set variable 'id' to value
 */
static void setvalue(void)
{
 int n;
 char **pp;
 
 for (n=0; list[n].idname; n++)
 {
  if (strcmp(list[n].idname, idname) == 0)
  {
   switch(list[n].type)
   {
    case 's':
     pp = (char**)list[n].ref;
     *pp = (char*)malloc(strlen(value)+1);
     if (*pp == 0)
     {
      fprintf(stderr, "**** fatal error: not enough memory ****\n");
      exit(3);
     }
     strcpy(*pp, value);
     break;

#if floating_point
    case 'd':
     *((double*)list[n].ref) = atof(value);
     break;
#endif

    case 'i':
     *((int*)list[n].ref) = atoi(value);
     break;

    case 'l':
     *((long*)list[n].ref) = atol(value);
     break;
   }
   return;
  }
 }
 error("unknown variable name");
}


/*
 * the scanning routine
 */
int cfg(char *name)
{
 FILE *fd;
 int state, c, index, deli;

 /* open file */
 filename = name;
 fd = fopen(name, "rb");
 if (fd == 0)
 {
  fprintf(stderr, "*** cannot find %s ***\n", name);
  return(-1);
 }

 /* read file */
 for (c=getc(fd), state=1, line=1; state!=0; c=getc(fd))
 {
  if (c == '\n') line++;
  if (c != EOF)
  {
   switch(state)
   {
    /* skip white space */
    case 1:
     if izspace(c) break;
     if (c == ';') {state = 2; break;}
     if (!izalpha(c))
     {
      error("variable name must begin with a letter");
      state = 2;
      break;
     }
     index = 0; idname[index++] = c; state = 3; break;

    /* go to next line */
    case 2:
     if (c != '\n') break;
     state = 1; break;

    /* variable name */
    case 3:
     if izspace(c) {idname[index]=0; state=4; break;}
     if (c == '=') {idname[index]=0; state=5; break;}
     if (!(izalnum(c) || (c=='_')))
      {error("illegal character in variable name"); state = 2; break;}
     idname[index++] = c;
     if (index >= MAXLEN)
      {error("variable name too long"); state = 2;}
     break;

    /* skip white space, expect '=' */
    case 4:
     if izspace(c) break;
     if (c == '=') {state=5; break;}
     error("'=' expected"); state = 2; break;

    /* skip white space, expect value */
    case 5:
     if izspace(c) break;
     index = 0;
     if (c == '\'' || c == '"') {deli = c; state = 7; break;}     
     ungetc(c, fd); state = 6; break;

    /* read value */
    case 6:
     if (izspace(c) || (c==';'))
     {
      value[index]=0;
      setvalue();
      state=2;
      break;
     }
     value[index++] = c;
     if (index >= MAXLEN)
      {error("variable value too long"); state = 2;}
     break;
   
    /* read quoted value */
    case 7:
     if (c == deli)
     {
      value[index]=0;
      setvalue();
      state=2;
      break;
     }
     if (c == '\n')
     {
      line--; error("unexpected end of line");
      line++; state=1; break;
     }
     value[index++] = c;
     if (index >= MAXLEN)
      {error("variable value too long"); state = 2;}
     break;
   }
  }
  else
  {
   if (state == 6)
   {
    ungetc(' ', fd);
   }
   else
   {
    if (state != 1 && state != 2) error("unexpected end of file");
    state = 0;
   }
  }
 }
 /* close file */
 fclose(fd);
 return 0;
}	