/* 
Reimplementation of the standard Alt-P hotkey for paragraph reflowing 
(formatting). 

  The function of Alt-P is changed for TeX files only, e.g. for files with 
one of the extensions  TEX LATEX STY CLS DTX. Then the paragraph reflowing 
stops at the first blank line (default) or at line containing:
  %	 --- comment 
  \begin --- new environment
  \end	 --- end of environment
  \item	 --- new item
  \[, \], \(, \) --- math environments

Installation: 
1. put  AltP4TeX.e  somewhere into your EPMPATH (I prefer \Apps\EPM\CSTeX)
2. add this line into your  Profile.erx  file:
'link AltP4TeX' /* Paragraph reflow (Alt-P hotkey) is TeX aware */
3. if you want to enhance its functionality, changes this source code. 
For instance, adding another TeX-reflow-stopping keyword, change the 
pm_reflow_is_tex_endline  procedure below. 

Note: a message appears if you are reflowing line block (Alt-L) in a TeX file. 
If you do not like this, then find the line with "InfoBox" below, comment 
it out and recompile (etpm AltP4TeX.e).

*** Some other comments ***

1. It could be changed so that line blocks work too: just go up with the 
   lower mark. 
2. This does not influence the text reflowing during typing. Maybe I should 
   ask Larry Margolis whether this is possible?

---

This macro will be/is distributed as a part of my pmCSTeX for EPM package 
for (em)(La)TeXing in EPM (see my homepage for more details, if you are 
not already a happy user of it). It can be freely used and distributed, 
but as usual: without any warranty. Please let me know if you find any 
improvement for this macro. 

Written by Petr Mikulik, 9. 12. 1997
mikulik@physics.muni.cz, http://www.sci.muni.cz/~mikulik/
*/

const
   E3      = EVERSION < 4

-- Set this to 1 if you like PE2's method of reflowing a paragraph -- moving
-- the cursor to the next paragraph.
compile if not defined(REFLOW_LIKE_PE)
REFLOW_LIKE_PE = 0
compile endif

include 'english.e'


def a_p=
   /* Protect the user from accidentally reflowing a marked  */
   /* area not in the current file, and give a good message. */
   mt = substr(marktype(), 1, 1)
   if mt='B' or mt='L' then
      getmark firstline,lastline,firstcol,lastcol,markfileid
      getfileid fileid
      if fileid<>markfileid then
         sayerror CANT_REFLOW__MSG'  'OTHER_FILE_MARKED__MSG
         return
      endif
   endif

   if mt<>' ' then
      if not check_mark_on_screen() then
compile if EVERSION < 5
         do while testkey()/==''; call getkey(); end
         sayerror MARK_OFF_SCRN_YN__MSG
         loop
            ch=upcase(getkey())
            if ch=YES_CHAR then sayerror 0; leave; endif
            if ch=NO_CHAR or ch=esc then sayerror 0; stop; endif
         endloop
compile else
         sayerror MARK_OFF_SCREEN__MSG
         stop
compile endif
      endif
   endif

   if mt='B' then
      'box r'
   elseif mt='C' then
      sayerror WRONG_MARK__MSG
   elseif mt='L' then
      if (wordpos(filetype(), 'TEX LATEX STY CLS DTX')) -- //PM
	then --//PM note: InfoBox is part of the pmCSTeX for EPM package
	  InfoBox('Warning','Reflowing line block may not be TeX safe. Use the Undo function (Ctrl/U) in case of problems. Edit and recompile AltP4TeX if you do not like this message.')
	endif
      reflow -- this is the default
   else  -- Standard text reflow split into a separate routine.
      if (wordpos(filetype(), 'TEX LATEX STY CLS DTX')) -- //PM
	then call pm_text_reflow()	-- TeX aware routine by PM
	else call text_reflow()		-- this is the default routine
	endif
   endif


-- reflowing paragraph in a TeX file
defproc pm_text_reflow
   if .line then
      getline line
      if line<>'' and -- If currently on a blank line, don't reflow.
	pm_reflow_is_tex_endline(line)=0  -- //PM neither if TeX nobreak line
	 then
         oldcursory=.cursory;oldcursorx=.cursorx; oldline=.line;oldcol=.col;
         unmark;mark_line
         call pm_pfind_blank_line()	-- //PM calls TeX aware routine
         -- Ver 3.11:  slightly revised test works better with GML sensitivity.
         if .line<>oldline then
            up
         else
            bottom
         endif
         mark_line
         reflow

compile if REFLOW_LIKE_PE   /* position on next paragraph (like PE) */
         down                       /* Thanks to Doug Short. */
         for i=.line+1 to .last
            getline line,i
            if line<>'' then i; leave; endif
         endfor
compile else
         /* or like old E */
         getmark firstline,lastline
         firstline
         .cursory=oldcursory;.cursorx=oldcursorx; oldline;.col=oldcol
compile endif
         unmark
      endif
   endif


defproc pm_pfind_blank_line
   -- Find first blank line after the current one.  Make that the new current
   -- line.  If no such line is found before the end of file, don't change the
   -- current line.
   for i = .line+1 to .last
      getline line,i
      -- //PM: away: -- Ver 3.11:  Modified to respect GML tags:  stop at first blank line
      -- //PM: away: -- or first line with a period or a colon (".:") in column 1.
      if line='' or -- //PM: away: not verify(substr(line,1,1), ".:" ) or
	pm_reflow_is_tex_endline(line)==1  -- //PM for being TeX aware
	then
         i
         leave
      endif
   endfor

-- //PM for being TeX aware
defproc pm_reflow_is_tex_endline(line)
if ( pos('%',line)>0 or pos('\begin',line)>0 or pos('\end',line)>0 or 
  pos('\item',line)>0 or pos('\[',line)>0 or pos('\]',line)>0 or 
  pos('\(',line)>0 or pos('\)',line)>0 or pos('\label',line)>0 )
  then return 1; endif
return 0

-- eof AltP4TeX.e
